﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessManager.EFR;
using System.Web.Security.AntiXss;

public partial class Referral : BasePage
{
    private const string GET_NUM_REFERRALS = "dbo.usp_EFR_GetReferralDateByPatientId";
    private const string GET_NUM_REFERRAL_ORIGINAL_RECORDS = "dbo.GetEFRReferralOriginalRecordCount";
    private const string GET_ORIGINAL_REFERRAL = "usp_EFR_GetOriginalReferral";
    private const string SAVE_ORIGINAL_REFERRAL = "usp_EFR_SaveOriginalReferral";
    private const string UPDATE_REFERRAL = "usp_EFR_UpdateWorkingReferral";

    private const string DOD_FRAGMENT_MESSAGE_MATCH = "This patient HAS a matching entry in the DoD fragment data extract file.";
    private const string DOD_FRAGMENT_MESSAGE_NOT_MATCH = "This patient DOES NOT HAVE a matching entry in the DoD fragment data extract file.";
    private const string DOD_FRAGMENT_DATE_MESSAGE = "DoD Fragment Data Received: ";

    private static IReadOnlyList<string> _navigationSources = new List<string>
    {
        "/Referrals.aspx",
        "/BioContactFollowups.aspx",
        "/EventReminders.aspx",
        "/Patient.aspx",
        "/PatientRelatedDiagnoses.aspx",
        "/PatientRelatedLabTests.aspx",
        "/PatientWorkflows.aspx",
        "/ReferralContact.aspx",
        "/VTAPatient.aspx"
    };

    protected readonly string _referralsPage = "referrals.aspx?statusId=";

    private readonly CityStateManager _cityStateManager;
    private readonly ReferralManager _referralManager;
    private readonly ReminderManager _reminderManager;
    private readonly WorkflowCaseManager _workflowCaseManager;
    private readonly DoDFragmentManager _dodFragmentManager;

    private UserAccountManager _userAccountManager;
    private EFR_REFERRAL _referral;
    private DataTable _dataTable;
    private bool _isOriginal = false;

    private enum WorkflowCaseStatus : int
    {
        New = 1,
        InProcess = 2,
        Completed = 3,
        Canceled = 4
    }

    public Referral()
    {
        _cityStateManager = new CityStateManager();
        _referralManager = new ReferralManager();
        _reminderManager = new ReminderManager();
        _workflowCaseManager = new WorkflowCaseManager();
        _dodFragmentManager = new DoDFragmentManager();
        _dataTable = new DataTable();

        //Manager Save Notification Event Handler
        _referralManager.BOSaveSuccess += manager_BOSaveSuccess;

        NavigationSources = _navigationSources;
    }

    protected void Page_Load(object sender, EventArgs e)
    {
        if (!IsPostBack)
        {
            // Determine if this is the Original Referral
            if (Request.QueryString["from"] == "workingReferral")
                _isOriginal = true;

            int refId;
            if (int.TryParse(Request.QueryString["referralId"], out refId))
            {
                if (refId == 0)
                {
                    _referral = new EFR_REFERRAL();
                    _referral.SetAsChangeTrackingRoot();
                    _referral.SetAsInsertOnSubmit();

                    AddContactButton.Enabled = false;
                }
                else if (!_isOriginal)
                {
                    getRefreshReferral(refId);
                }
                else
                {
                    // load original data

                    _referral = new EFR_REFERRAL();
                    getOriginalReferral(refId);
                }
            }

            SetupPatientNameControl();
            SetupReferralDetailControls();
            SetupDodFragmentMessage(_referral);
            //load dropdown data on initial load
            RefreshVAMCListAndSelectInstitution();

            if (_referral.REFERRAL_ID > 0)
            {
                LoadIraqAfghanServiceHealthFactor(_referral.PATIENT_ID);

                ReLoadInstitutionAddress(true);

                ShowReferralFromEntity();

                CheckReferralForDuplicates();

                CheckAndSetupOriginalReferral();

                RefreshClinicalReminderListAndSelectReminder(_referral);

                CheckAndSetupVTABButton();

                // Handle Original Referral 

                SetupOriginalReferralLink();

            }
            else
            {
                SetupManualReferral();
            }

            SetReferringPage();
        }

        CancelButton.Text = "Back";

        switch (_referral.STD_REFERRALSTS_ID)
        {
            case 4:
                SetUpNewReferral();
                break;
            case 5:
                SetupInProcessReferral();
                break;
            case 6:
                SetupInProcessReferral();
                break;
            case 10:
                SetUpNoActionReferral();
                break;
            case 9:
                SetUpDeclinedReferral();
                break;
            default:
                SetUpNewReferral();
                break;
        }
    }

    private void SetupDodFragmentMessage(EFR_REFERRAL referral)
    {
        // referral status 4 = New, 5 = Open, 6 = closed
        if (referral.STD_REFERRALSTS_ID == 4
            || referral.STD_REFERRALSTS_ID == 5
            || referral.STD_REFERRALSTS_ID == 6)
        {
            LabelDoDFragmentMessage.Visible = true;
            LabelDoDFragmentDate.Visible = true;

            DoD_FRAGMENT_DATA_RECEIVELOG fragment = _dodFragmentManager.SelectBySSN(referral.PATIENT.Snum);
            System.Nullable<DateTime> extractDate = _dodFragmentManager.GetLastExtractDate(referral.PATIENT.Snum);

            if (fragment != null)
            {
                LabelDoDFragmentMessage.Text = DOD_FRAGMENT_MESSAGE_MATCH + "<br /><br />";
            }
            else
            {
                LabelDoDFragmentMessage.Text = DOD_FRAGMENT_MESSAGE_NOT_MATCH + "<br /><br />";
            }

            LabelDoDFragmentDate.Text = DOD_FRAGMENT_DATE_MESSAGE;
            if (extractDate != null)
            {
                LabelDoDFragmentDate.Text = DOD_FRAGMENT_DATE_MESSAGE + String.Format("{0:d}", extractDate);
            }
            else
            {
                LabelDoDFragmentDate.Text = DOD_FRAGMENT_DATE_MESSAGE + "N/A";
            }
        }
        else
        {
            LabelDoDFragmentMessage.Visible = false;
            LabelDoDFragmentDate.Visible = false;
        }
    }

    private void RefreshClinicalReminderListAndSelectReminder(EFR_REFERRAL referral)
    {
        LabelClinicalReminder.Text = "";

        var referralDetails = _referral.REFERRAL_DETAILs;
        bool injuredByBulletQ1 = false;
        bool blastExplosionInjuryQ2 = false;
        bool fragmentsRemovedInSurgeryQ3 = false;
        bool fragmentsInBodyQ4 = false;
        bool fragmentsOnRadiographQ4a = false;
        bool unableToScreen = false;
        bool healthFactorPresent = false;
        const int UnableToScreenId = 13;
        const int NoHealthFactorsPresent = 14;
        bool riskCatetoryIsNoHealthFactorPresent = false;

        if (!_referral.CREATEDBY.Contains("ETL Process"))
        {
            List<REFERRAL_DETAIL> healthFactors = (from e in _referral.REFERRAL_DETAILs
                                                   where e.STD_HEALTHFACTOR_ID == 3
                                                      || e.STD_HEALTHFACTOR_ID == 4
                                                      || e.STD_HEALTHFACTOR_ID == 20
                                                      || e.STD_HEALTHFACTOR_ID == 21
                                                   select e).ToList();

            if (healthFactors.Count == 0)
            {
                _referral.STD_REMINDERCLASS = _reminderManager.getReminderByReminderID(NoHealthFactorsPresent);
                LabelClinicalReminder.Text = _referral.STD_REMINDERCLASS.DESCRIPTION_TEXT;
                riskCatetoryIsNoHealthFactorPresent = true;
            }
        }

        if (riskCatetoryIsNoHealthFactorPresent == false)
        {
            foreach (var referralDetail in referralDetails)
            {
                switch (referralDetail.STD_HEALTHFACTOR_ID)
                {
                    case 4:
                        injuredByBulletQ1 = true;
                        break;
                    case 3:
                        blastExplosionInjuryQ2 = true;
                        break;
                    case 13:
                        fragmentsRemovedInSurgeryQ3 = true;
                        break;
                    case 10:
                        fragmentsRemovedInSurgeryQ3 = false;
                        break;
                    case 29:
                        fragmentsRemovedInSurgeryQ3 = false;
                        break;
                    case 9:
                        fragmentsInBodyQ4 = true;
                        break;
                    case 22:
                        fragmentsInBodyQ4 = false;
                        break;
                    case 27:
                        fragmentsInBodyQ4 = false;
                        break;
                    case 12:
                        fragmentsOnRadiographQ4a = true;
                        break;
                    case 23:
                        fragmentsOnRadiographQ4a = false;
                        break;
                    case 28:
                        fragmentsOnRadiographQ4a = false;
                        break;
                    case 25:
                        unableToScreen = true;
                        break;
                    case 75:
                        unableToScreen = true;
                        break;
                    case 79:
                        unableToScreen = true;
                        break;

                    default:
                        break;
                }

                switch (referralDetail.STD_HEALTHFACTOR_ID)
                {
                    case 4:
                    case 21:
                    case 3:
                    case 20:
                    case 18:
                    case 24:
                    case 17:
                    case 26:
                    case 19:
                    case 16:
                    case 8:
                    case 15:
                    case 2:
                    case 1:
                    case 13:
                    case 10:
                    case 29:
                    case 14:
                    case 11:
                    case 30:
                    case 9:
                    case 22:
                    case 27:
                    case 12:
                    case 23:
                    case 28:
                        healthFactorPresent = true;
                        break;
                    default:
                        break;
                }

            }

            //the patient has had a fragment removed at surgery or has a documented retained fragment
            if ((injuredByBulletQ1 || blastExplosionInjuryQ2)
                && (fragmentsRemovedInSurgeryQ3 || (fragmentsInBodyQ4 && fragmentsOnRadiographQ4a)))
            {
                _referral.STD_REMINDERCLASS = _reminderManager.getReminderByReminderID(2);
                LabelClinicalReminder.Text = _referral.STD_REMINDERCLASS.DESCRIPTION_TEXT;
            }

            //the patient has a high likelihood of having a retained fragment
            if ((injuredByBulletQ1 || blastExplosionInjuryQ2)
                && !fragmentsRemovedInSurgeryQ3
                && (fragmentsInBodyQ4 && !fragmentsOnRadiographQ4a))
            {
                _referral.STD_REMINDERCLASS = _reminderManager.getReminderByReminderID(7);
                LabelClinicalReminder.Text = _referral.STD_REMINDERCLASS.DESCRIPTION_TEXT;
            }

            //the veteran possibly has a retained fragment as a result of injuries sustained while serving in the area of conflict
            if ((injuredByBulletQ1 || blastExplosionInjuryQ2)
                && !fragmentsRemovedInSurgeryQ3
                && !fragmentsInBodyQ4)
            {
                _referral.STD_REMINDERCLASS = _reminderManager.getReminderByReminderID(9);
                LabelClinicalReminder.Text = _referral.STD_REMINDERCLASS.DESCRIPTION_TEXT;
            }

            //Unable to Screen
            if (unableToScreen && !healthFactorPresent)
            {
                _referral.STD_REMINDERCLASS = _reminderManager.getReminderByReminderID(UnableToScreenId);
                LabelClinicalReminder.Text = _referral.STD_REMINDERCLASS.DESCRIPTION_TEXT;
            }
            //the veteran likely does not have a retained fragment as a result of injuries sustained while serving in the area of conflict
            else if (!injuredByBulletQ1
                && !blastExplosionInjuryQ2)
            {
                _referral.STD_REMINDERCLASS = _reminderManager.getReminderByReminderID(12);
                LabelClinicalReminder.Text = _referral.STD_REMINDERCLASS.DESCRIPTION_TEXT;
            }
        }
    }

    protected override void LoadViewState(object savedState)
    {
        base.LoadViewState(savedState);
        string referralXml = (string)ViewState["referralXml"];
        _referral = SerializerUtil.Deserialize<EFR_REFERRAL>(referralXml);
    }

    protected override object SaveViewState()
    {
        ViewState["referralXml"] = _referral.Serialize();
        return base.SaveViewState();
    }

    protected void VTADataButton_Click(object sender, EventArgs e)
    {

        Response.Redirect("VTAPatient.aspx?id={0}&patientId={1}", _referral.REFERRAL_ID, _referral.PATIENT_ID);

    }
    protected void OKButton_Click(object sender, EventArgs e)
    {
        int statusId;

        //set appropriate status based on user selection or no selection at all 
        if (First.Checked)
            statusId = 5;

        else if (Second.Checked)
            statusId = 9;

        else if (Third.Checked)
            statusId = 10;
        else
            statusId = _referral.STD_REFERRALSTS_ID;

        //referral.SetAsChangeTrackingRoot();


        //see if status has changed and do some operations if it does
        if (statusId != _referral.STD_REFERRALSTS_ID)
        {
            _referral.STD_REFERRALSTS_ID = statusId;

            _referral.StatusChanged = true;
            switch (statusId)
            {
                case 5:

                    //create each type requested workflow and assign some properties
                    foreach (ListItem item in WorkflowType.Items)
                    {
                        if (item.Selected)
                        {
                            WKF_CASE wfCase = new WKF_CASE();

                            wfCase.REFERRAL_ID = _referral.REFERRAL_ID;

                            int patientId;

                            if (int.TryParse(FormViewPatientInformation.DataKey.Value.ToString(), out patientId))
                            {
                                wfCase.PATIENT_ID = patientId;
                            }

                            //this is EFR workflow
                            int caseType;

                            if (int.TryParse(item.Value, out caseType))
                                wfCase.STD_WKFCASETYPE_ID = caseType;

                            wfCase.CASE_START_DATE = DateTime.Today;
                            //this is an in-process workflow

                            wfCase.STD_WKFCASESTS_ID = (int)Enum.Parse(typeof(WorkflowCaseStatus), Enum.GetName(typeof(WorkflowCaseStatus), WorkflowCaseStatus.InProcess));
                            wfCase.CASE_NUMBER = String.Empty;

                            //now add newly generated workflow to it's base referral 
                            _referral.WKF_CASEs.Add(wfCase);

                        }
                    }

                    //clear previous reasons if any present 
                    _referral.COMMENT_TEXT = null;

                    SetupInProcessReferral();
                    break;

                case 6:

                    _referral.COMMENT_TEXT = null;

                    SetupInProcessReferral();
                    break;

                case 9:

                    _referral.COMMENT_TEXT = TextIneligibleReason.Text;
                    SetUpDeclinedReferral();
                    break;

                case 10:

                    SetUpNoActionReferral();
                    break;

                default:
                    break;
            }

            FormViewBreadCrumb.DataBind();
        }

        //get modified values and do the update
        SaveReferralFromPage();

        SaveHealthFactorsFromPage();

        try
        {
            // Prevent duplicate insertions of new records
            if (_referral.REFERRAL_ID == 0)
            {
                if (setUpReferralLock(_referral.PATIENT_ID))
                {
                    SaveReferral();
                }
                else
                {
                    string message = "";

                    message = "Another user has already or is currently attempting to create this record";
                    OKButton.Enabled = false;
                    OKButtonValidator.IsValid = false;
                    OKButtonValidator.ErrorMessage = message;
                }
            }
            else
            {
                SaveReferral();
            }
        }
        catch (Exception ex)
        {
            Logging.WriteLogToFile("Error processing new referral", ex);
            OKButtonValidator.IsValid = false;
            OKButtonValidator.ErrorMessage = "There was an error processing your request: " + ex.Message;
        }
    }

    protected void CancelButton_Click(object sender, EventArgs e)
    {
        string currentQueue = "4";

        if (!string.IsNullOrEmpty(Helpers.GetStringFromObject(Request.QueryString["statusId"])))
        {
            currentQueue = Helpers.GetStringFromObject(Request.QueryString["statusId"]);
        }
        else if (_referral != null)
        {
            currentQueue = _referral.STD_REFERRALSTS_ID.ToString();
        }

        Uri from = ViewState["referrer"] as Uri;
        if (UriUtility.IsValidPathSource(NavigationSources, from) 
            && !from.LocalPath.Equals("/Referrals.aspx", StringComparison.OrdinalIgnoreCase))
        {
            RedirectToReferringPage();
        }
        else if (currentQueue == "9999")
        {
            Response.Redirect("BioContactFollowups.aspx");
        }
        else
        {
            ResponseRedirect(_referralsPage, currentQueue);
        }
    }

    private void SaveReferral()
    {
        bool newReferral = false;

        if (_referral.REFERRAL_ID == 0)
            newReferral = true;

        try
        {
            _referralManager.Update(_referral);
        }
        catch (Exception e)
        {
            Logging.WriteLogToFile("Error processing new referral", e);
        }
        finally
        {
            if (Globals.NEW_PENDING_REFERRALS.Contains(_referral.PATIENT_ID))
                Globals.NEW_PENDING_REFERRALS.Remove(_referral.PATIENT_ID);
        }

        if (newReferral)
        {
            try
            {
                // Check to make sure this entry doesn't exist 
                var succeeded = SqlProvider.ExecuteScalar(
                    "RegistryConnectionString", SAVE_ORIGINAL_REFERRAL, new object[] { _referral.REFERRAL_ID });

                if (succeeded != DBNull.Value && Convert.ToInt32(succeeded) == 0)
                {
                    Logging.WriteLogToFile(CurrentUser.UserId.ToString() + " saved Original referral: " + _referral.REFERRAL_ID.ToString());
                }
                else
                {
                    Logging.WriteLogToFile(CurrentUser.UserId.ToString() + " Failed saving original referral: " + _referral.REFERRAL_ID.ToString()
                        + " - Error Code: " + succeeded);
                }
            }
            catch (Exception e)
            {
                Logging.WriteLogToFile("Error saving original referral", e);
            }
        }

        if (Request.QueryString["referralId"] == "0")
        {
            _redirectAfterSave = string.Format("referral.aspx?referralId={0}", _referral.REFERRAL_ID);
        }
        else
        {
            getRefreshReferral(_referral.REFERRAL_ID);
            ShowReferralFromEntity();
            ReBindChangedData();
            RefreshClinicalReminderListAndSelectReminder(_referral);
        }

        //(this.Master as RegistryMaster).SetDirtyFlag(false);
        var master = this.Master as RegistryMaster;
        if (master != null)
        {
            master.SetDirtyFlag(false);
        }

    }

    private bool setUpReferralLock(int patientId)
    {
        // This creates a 'lock' so duplicate new referrals cannot be created

        bool retValue = true;

        if (Globals.NEW_PENDING_REFERRALS != null && Globals.NEW_PENDING_REFERRALS.Contains(patientId))
        {
            retValue = false;
        }
        else
        {
            Globals.NEW_PENDING_REFERRALS.Add(_referral.PATIENT_ID);

            // Check to make sure this entry doesn't exist 
            var referralDate = SqlProvider.ExecuteScalar(
                "RegistryConnectionString", GET_NUM_REFERRALS, new object[] { _referral.PATIENT_ID });

            if (referralDate != DBNull.Value)
            {
                retValue = false;
                Globals.NEW_PENDING_REFERRALS.Remove(_referral.PATIENT_ID);
            }
        }

        return retValue;
    }

    private void SetupManualReferral()
    {
        VTADataButton.Visible = false;
    }

    private void SetupReferralDetailControls()
    {
        bool manualReferral = (_referral.REFERRAL_ID == 0);

        if (_referral.REFERRAL_ID > 0)
        {
            UsersManager userManager = new UsersManager();
            string createdBy = _referral.CREATEDBY;
            if (createdBy.IndexOf("\\") > 0)
            {
                createdBy = createdBy.Split('\\')[1];
            }
            USER user = userManager.SelectByUsername(createdBy);
            if (user != null)
            {
                manualReferral = true;
            }
        }

        if (manualReferral)
        {
            TEFSCDateForLabel.Visible = false;
            TEFSCDateLabel.Visible = false;
            ReviewedDateLabel.Visible = false;
            ReviewedDateBox.Visible = false;
            ReviewedByLabel.Visible = false;
            ReviewedByBox.Visible = false;
            LabelCreatedBy.Text = CurrentUser.UserName;
            LabelReferralDate.Text = String.Format("{0:d}", DateTime.Now);
        }

    }

    private void SetupPatientNameControl()
    {
        Label nameLabel = (Label)FormViewPatientInformation.FindControl("NameLabel");
        if (nameLabel != null)
        {
            if (_referral.REFERRAL_ID > 0)
            {
                nameLabel.Visible = false;
            }
            else
            {
                nameLabel.Visible = true;
            }

        }

        HyperLink linkNameLabel = (HyperLink)FormViewPatientInformation.FindControl("PatientWorkflowsLink");
        if (linkNameLabel != null)
        {
            if (_referral.REFERRAL_ID > 0)
            {
                linkNameLabel.Visible = true;
            }
            else
            {
                linkNameLabel.Visible = false;
            }
        }
    }

    private void SetUpNewReferral()
    {
        PanelChecklist.Visible = true;
        PanelWorkflows.Visible = false;

        PanelNoActionIneligible.Visible = false;

    }

    private void SetUpNoActionReferral()
    {
        PanelChecklist.Visible = false;
        PanelWorkflows.Visible = false;

        PanelNoActionIneligible.Visible = true;

        IneligibilityReason.Text = "No action Required";

    }

    private void SetupInProcessReferral()
    {

        PanelChecklist.Visible = false;
        PanelWorkflows.Visible = true;

        PanelNoActionIneligible.Visible = false;



    }

    //referral was declined so no workflows are visible, only declined reason
    private void SetUpDeclinedReferral()
    {
        PanelChecklist.Visible = false;
        PanelWorkflows.Visible = false;

        PanelNoActionIneligible.Visible = true;

        IneligibilityReason.Text = "Ineligible" + "&nbsp;".Repeat(30) + "Reason: " + AntiXssEncoder.HtmlEncode(_referral.COMMENT_TEXT, false);

    }


    private void SetUpClosedReferral()
    {
    }

    private void SetupOriginalReferralLink()
    {
        var numReferrals = SqlProvider.ExecuteScalar(
                        "RegistryConnectionString", GET_NUM_REFERRAL_ORIGINAL_RECORDS, new object[] { _referral.REFERRAL_ID });

        if (numReferrals != DBNull.Value && numReferrals != null)
        {
            int iNumReferrals = 0;

            if (int.TryParse(numReferrals.ToString(), out iNumReferrals))
            {
                if (iNumReferrals >= 1 && !_isOriginal)
                {
                    hlOriginalReferral.Visible = true;

                    hlOriginalReferral.NavigateUrl =
                        HttpContext.Current.Request.Url.AbsoluteUri.Substring(0,
                            HttpContext.Current.Request.Url.AbsoluteUri.IndexOf('?')) +
                            "?referralId=" + _referral.REFERRAL_ID + "&from=workingReferral";
                }
            }
            else
            {
                Logging.WriteLogToFile(User.Identity.Name, "numReferrals is not an int: " + numReferrals.ToString());
            }
        }
        else
        {
            Logging.WriteLogToFile(User.Identity.Name, "numReferrals is null");
        }
    }

    private void getRefreshReferral(int rId)
    {
        _referral = _referralManager.getReferralByReferralID(rId);
        _referral.SetAsChangeTrackingRoot();

        if (_referral.REFERRAL_DETAILs.GroupBy(x => x.STD_HEALTHFACTOR_ID).Any(s => s.Count() > 1))
        {
            DuplicateHealthfactorIndicator.Value = "1";

        }
        else
        {
            DuplicateHealthfactorIndicator.Value = "0";

        }

    }

    private void getOriginalReferral(int rId)
    {
        try
        {
            _dataTable = SqlProvider.ExecuteSPDataTable(
                "RegistryConnectionString", GET_ORIGINAL_REFERRAL,
                new object[] { rId });

            _referral.REFERRAL_ID = Convert.ToInt32(_dataTable.Rows[0]["REFERRAL_ID"].ToString());

            if (_dataTable.Rows[0]["STD_INSTITUTION_ID"].ToString().Length > 0)
                _referral.STD_INSTITUTION_ID = Convert.ToInt32(_dataTable.Rows[0]["STD_INSTITUTION_ID"].ToString());

            _referral.ADDRESS_LINE1 = _dataTable.Rows[0]["ADDRESS_LINE1"].ToString();
            _referral.ADDRESS_LINE2 = _dataTable.Rows[0]["ADDRESS_LINE2"].ToString();
            _referral.ADDRESS_LINE3 = _dataTable.Rows[0]["ADDRESS_LINE3"].ToString();
            _referral.CITY = _dataTable.Rows[0]["CITY"].ToString();
            _referral.POSTAL_CODE = _dataTable.Rows[0]["POSTAL_CODE"].ToString();
            _referral.STATE = _dataTable.Rows[0]["STATE"].ToString();
            _referral.COUNTRY = _dataTable.Rows[0]["COUNTRY"].ToString();

            _referral.COMMENT_TEXT = _dataTable.Rows[0]["COMMENT_TEXT"].ToString();

            if (_dataTable.Rows[0]["REFERRAL_DATE"].ToString().Length > 0)
                _referral.REFERRAL_DATE = Convert.ToDateTime(_dataTable.Rows[0]["REFERRAL_DATE"]);

            _referral.CREATED = Convert.ToDateTime(_dataTable.Rows[0]["CREATED"]);
            _referral.CREATEDBY = _dataTable.Rows[0]["CREATEDBY"].ToString();
            _referral.UPDATED = Convert.ToDateTime(_dataTable.Rows[0]["UPDATED"]);
            _referral.UPDATEDBY = _dataTable.Rows[0]["UPDATEDBY"].ToString();
            _referral.REVIEW_BY = _dataTable.Rows[0]["REVIEW_BY"].ToString();
            if (_dataTable.Rows[0]["REVIEW_DATE"].ToString().Length > 0)
                _referral.REVIEW_DATE = Convert.ToDateTime(_dataTable.Rows[0]["REVIEW_DATE"]);

            _referral.DUPLICATE_FLAG = Convert.ToBoolean(_dataTable.Rows[0]["DUPLICATE_FLAG"]);
            _referral.PATIENT_ID = Convert.ToInt32(_dataTable.Rows[0]["PATIENT_ID"]);

            _referral.PATIENT = new PATIENT();
            _referral.PATIENT.Snum = _dataTable.Rows[0]["SSN"].ToString();

            _referral.ORDERBY_PROVIDER_ADDRESS_LINE1 = _dataTable.Rows[0]["ORDERBY_PROVIDER_ADDRESS_LINE1"].ToString();
            _referral.ORDERBY_PROVIDER_ADDRESS_LINE2 = _dataTable.Rows[0]["ORDERBY_PROVIDER_ADDRESS_LINE2"].ToString();
            _referral.ORDERBY_PROVIDER_ADDRESS_LINE3 = _dataTable.Rows[0]["ORDERBY_PROVIDER_ADDRESS_LINE3"].ToString();
            _referral.ORDERBY_PROVIDER_CITY = _dataTable.Rows[0]["ORDERBY_PROVIDER_CITY"].ToString();
            _referral.ORDERBY_PROVIDER_COUNTRY = _dataTable.Rows[0]["ORDERBY_PROVIDER_COUNTRY"].ToString();
            _referral.ORDERBY_PROVIDER_EMAIL = _dataTable.Rows[0]["ORDERBY_PROVIDER_EMAIL"].ToString();
            _referral.ORDERBY_PROVIDER_FAX = _dataTable.Rows[0]["ORDERBY_PROVIDER_FAX"].ToString();
            _referral.ORDERBY_PROVIDER_ID_NUMBER = _dataTable.Rows[0]["ORDERBY_PROVIDER_ID_NUMBER"].ToString();
            _referral.ORDERBY_PROVIDER_NAME = _dataTable.Rows[0]["ORDERBY_PROVIDER_NAME"].ToString();
            _referral.ORDERBY_PROVIDER_PHONE = _dataTable.Rows[0]["ORDERBY_PROVIDER_PHONE"].ToString();
            _referral.ORDERBY_PROVIDER_POSTAL_CODE = _dataTable.Rows[0]["ORDERBY_PROVIDER_POSTAL_CODE"].ToString();
            _referral.ORDERBY_PROVIDER_STATE = _dataTable.Rows[0]["ORDERBY_PROVIDER_STATE"].ToString();
            _referral.ORDERBY_PROVIDER_ZIP_PLUS_4 = _dataTable.Rows[0]["ORDERBY_PROVIDER_ZIP_PLUS_4"].ToString();

            _referral.PROVIDER_EMAIL = _dataTable.Rows[0]["PROVIDER_EMAIL"].ToString();
            _referral.PROVIDER_ADDRESS_LINE1 = _dataTable.Rows[0]["PROVIDER_ADDRESS_LINE1"].ToString();
            _referral.PROVIDER_ADDRESS_LINE2 = _dataTable.Rows[0]["PROVIDER_ADDRESS_LINE2"].ToString();
            _referral.PROVIDER_ADDRESS_LINE3 = _dataTable.Rows[0]["PROVIDER_ADDRESS_LINE3"].ToString();
            _referral.PROVIDER_CITY = _dataTable.Rows[0]["PROVIDER_CITY"].ToString();
            _referral.PROVIDER_COUNTRY = _dataTable.Rows[0]["PROVIDER_COUNTRY"].ToString();
            _referral.PROVIDER_COUNTY = _dataTable.Rows[0]["PROVIDER_COUNTY"].ToString();
            _referral.PROVIDER_EMAIL = _dataTable.Rows[0]["PROVIDER_EMAIL"].ToString();
            _referral.PROVIDER_FAX = _dataTable.Rows[0]["PROVIDER_FAX"].ToString();
            _referral.PROVIDER_ID_NUMBER = _dataTable.Rows[0]["PROVIDER_ID_NUMBER"].ToString();
            _referral.PROVIDER_NAME = _dataTable.Rows[0]["PROVIDER_NAME"].ToString();
            _referral.PROVIDER_PHONE = _dataTable.Rows[0]["PROVIDER_PHONE"].ToString();
            _referral.PROVIDER_POSTAL_CODE = _dataTable.Rows[0]["PROVIDER_POSTAL_CODE"].ToString();
            _referral.PROVIDER_STATE = _dataTable.Rows[0]["PROVIDER_STATE"].ToString();
            _referral.PROVIDER_ZIP_PLUS_4 = _dataTable.Rows[0]["PROVIDER_ZIP_PLUS_4"].ToString();

            _referral.VAMC_CONTACT_EMAIL = _dataTable.Rows[0]["VAMC_CONTACT_EMAIL"].ToString();
            _referral.VAMC_CONTACT_FAX = _dataTable.Rows[0]["VAMC_CONTACT_FAX"].ToString();
            _referral.VAMC_CONTACT_NAME = _dataTable.Rows[0]["VAMC_CONTACT_NAME"].ToString();
            _referral.VAMC_CONTACT_PHONE = _dataTable.Rows[0]["VAMC_CONTACT_PHONE"].ToString();

            _referral.REFERRAL_CLASS_TEXT = _dataTable.Rows[0]["REFERRAL_CLASS_TEXT"].ToString();
            if (_dataTable.Rows[0]["STD_REFERRALRANK_ID"].ToString().Length > 0)
                _referral.STD_REFERRALRANK_ID = Convert.ToInt32(_dataTable.Rows[0]["STD_REFERRALRANK_ID"]);
            _referral.STD_REFERRALSTS_ID = Convert.ToInt32(_dataTable.Rows[0]["STD_REFERRALSTS_ID"]);
            _referral.STD_REGISTRY_ID = Convert.ToInt32(_dataTable.Rows[0]["STD_REGISTRY_ID"]);
            if (_dataTable.Rows[0]["STD_REMINDERCLASS_ID"].ToString().Length > 0)
                _referral.STD_REMINDERCLASS_ID = Convert.ToInt32(_dataTable.Rows[0]["STD_REMINDERCLASS_ID"]);

            _referral.STD_INSTITUTION = new STD_INSTITUTION();
            _referral.STD_INSTITUTION.ID = Convert.ToInt32(_dataTable.Rows[0]["ID"].ToString());
            _referral.STD_INSTITUTION.STREETADDRESSLINE1 = _dataTable.Rows[0]["STREETADDRESSLINE1"].ToString();
            _referral.STD_INSTITUTION.STREETADDRESSLINE2 = _dataTable.Rows[0]["STREETADDRESSLINE2"].ToString();
            _referral.STD_INSTITUTION.STREETADDRESSLINE3 = _dataTable.Rows[0]["STREETADDRESSLINE3"].ToString();
            _referral.STD_INSTITUTION.STREETCITY = _dataTable.Rows[0]["STREETCITY"].ToString();
            _referral.STD_INSTITUTION.STREETPOSTALCODE = _dataTable.Rows[0]["STREETPOSTALCODE"].ToString();
            //if (_dataTable.Rows[0]["STD_INSTITUTION_ID"].ToString().Length > 0)
            //    _referral.STD_INSTITUTION_ID = Convert.ToInt32(_dataTable.Rows[0]["STD_INSTITUTION_ID"]);

            if (_dataTable.Rows[0]["STREETSTATE_ID"].ToString().Length > 0)
                _referral.STD_INSTITUTION.STREETSTATE_ID = Convert.ToInt32(_dataTable.Rows[0]["STREETSTATE_ID"].ToString());
            if (_dataTable.Rows[0]["STREETCOUNTRY_ID"].ToString().Length > 0)
                _referral.STD_INSTITUTION.STREETCOUNTRY_ID = Convert.ToInt32(_dataTable.Rows[0]["STREETCOUNTRY_ID"].ToString());

            _referral.STD_INSTITUTION.STREETADDRESSLINE1 = _dataTable.Rows[0]["MAILINGADDRESSLINE1"].ToString();
            _referral.STD_INSTITUTION.STREETADDRESSLINE2 = _dataTable.Rows[0]["MAILINGADDRESSLINE2"].ToString();
            _referral.STD_INSTITUTION.STREETADDRESSLINE3 = _dataTable.Rows[0]["MAILINGADDRESSLINE3"].ToString();
            _referral.STD_INSTITUTION.STREETCITY = _dataTable.Rows[0]["MAILINGCITY"].ToString();
            _referral.STD_INSTITUTION.STREETPOSTALCODE = _dataTable.Rows[0]["MAILINGPOSTALCODE"].ToString();

            if (_dataTable.Rows[0]["MAILINGSTATE_ID"].ToString().Length > 0)
                _referral.STD_INSTITUTION.MAILINGSTATE_ID = Convert.ToInt32(_dataTable.Rows[0]["MAILINGSTATE_ID"].ToString());
            if (_dataTable.Rows[0]["MAILINGCOUNTRY_ID"].ToString().Length > 0)
                _referral.STD_INSTITUTION.MAILINGCOUNTRY_ID = Convert.ToInt32(_dataTable.Rows[0]["MAILINGCOUNTRY_ID"].ToString());

            // Get Health Factors into the referral
            _referral.REFERRAL_DETAILs = new System.Data.Linq.EntitySet<REFERRAL_DETAIL>();

            if (_dataTable.Rows[0]["EFP"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_69 = new REFERRAL_DETAIL();
                rd_69.STD_HEALTHFACTOR_ID = 69;
                _referral.REFERRAL_DETAILs.Add(rd_69);
            }
            if (_dataTable.Rows[0]["NEF"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_70 = new REFERRAL_DETAIL();
                rd_70.STD_HEALTHFACTOR_ID = 70;
                _referral.REFERRAL_DETAILs.Add(rd_70);
            }
            if (_dataTable.Rows[0]["BI"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_4 = new REFERRAL_DETAIL();
                rd_4.STD_HEALTHFACTOR_ID = 4;
                _referral.REFERRAL_DETAILs.Add(rd_4);
            }
            if (_dataTable.Rows[0]["NBI"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_21 = new REFERRAL_DETAIL();
                rd_21.STD_HEALTHFACTOR_ID = 21;
                _referral.REFERRAL_DETAILs.Add(rd_21);
            }
            if (_dataTable.Rows[0]["BEI"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_3 = new REFERRAL_DETAIL();
                rd_3.STD_HEALTHFACTOR_ID = 3;
                _referral.REFERRAL_DETAILs.Add(rd_3);
            }
            if (_dataTable.Rows[0]["NBEI"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_20 = new REFERRAL_DETAIL();
                rd_20.STD_HEALTHFACTOR_ID = 20;
                _referral.REFERRAL_DETAILs.Add(rd_20);
            }
            if (_dataTable.Rows[0]["IED"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_17 = new REFERRAL_DETAIL();
                rd_17.STD_HEALTHFACTOR_ID = 17;
                _referral.REFERRAL_DETAILs.Add(rd_17);
            }
            if (_dataTable.Rows[0]["RPG"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_26 = new REFERRAL_DETAIL();
                rd_26.STD_HEALTHFACTOR_ID = 26;
                _referral.REFERRAL_DETAILs.Add(rd_26);
            }
            if (_dataTable.Rows[0]["LM"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_19 = new REFERRAL_DETAIL();
                rd_19.STD_HEALTHFACTOR_ID = 19;
                _referral.REFERRAL_DETAILs.Add(rd_19);
            }
            if (_dataTable.Rows[0]["Gr"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_16 = new REFERRAL_DETAIL();
                rd_16.STD_HEALTHFACTOR_ID = 16;
                _referral.REFERRAL_DETAILs.Add(rd_16);
            }
            if (_dataTable.Rows[0]["EF"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_8 = new REFERRAL_DETAIL();
                rd_8.STD_HEALTHFACTOR_ID = 8;
                _referral.REFERRAL_DETAILs.Add(rd_8);
            }
            if (_dataTable.Rows[0]["FF"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_15 = new REFERRAL_DETAIL();
                rd_15.STD_HEALTHFACTOR_ID = 15;
                _referral.REFERRAL_DETAILs.Add(rd_15);
            }
            if (_dataTable.Rows[0]["UT"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_2 = new REFERRAL_DETAIL();
                rd_2.STD_HEALTHFACTOR_ID = 2;
                _referral.REFERRAL_DETAILs.Add(rd_2);
            }
            if (_dataTable.Rows[0]["Other"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_1 = new REFERRAL_DETAIL();
                rd_1.STD_HEALTHFACTOR_ID = 1;
                _referral.REFERRAL_DETAILs.Add(rd_1);
            }
            if (_dataTable.Rows[0]["FIB"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_9 = new REFERRAL_DETAIL();
                rd_9.STD_HEALTHFACTOR_ID = 9;
                _referral.REFERRAL_DETAILs.Add(rd_9);
            }
            if (_dataTable.Rows[0]["NFIB"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_22 = new REFERRAL_DETAIL();
                rd_22.STD_HEALTHFACTOR_ID = 22;
                _referral.REFERRAL_DETAILs.Add(rd_22);
            }
            if (_dataTable.Rows[0]["UFIB"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_27 = new REFERRAL_DETAIL();
                rd_27.STD_HEALTHFACTOR_ID = 27;
                _referral.REFERRAL_DETAILs.Add(rd_27);
            }
            if (_dataTable.Rows[0]["FOR"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_12 = new REFERRAL_DETAIL();
                rd_12.STD_HEALTHFACTOR_ID = 12;
                _referral.REFERRAL_DETAILs.Add(rd_12);
            }
            if (_dataTable.Rows[0]["NFOR"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_23 = new REFERRAL_DETAIL();
                rd_23.STD_HEALTHFACTOR_ID = 23;
                _referral.REFERRAL_DETAILs.Add(rd_23);
            }
            if (_dataTable.Rows[0]["UFOR"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_28 = new REFERRAL_DETAIL();
                rd_28.STD_HEALTHFACTOR_ID = 28;
                _referral.REFERRAL_DETAILs.Add(rd_28);
            }
            if (_dataTable.Rows[0]["IV"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_18 = new REFERRAL_DETAIL();
                rd_18.STD_HEALTHFACTOR_ID = 18;
                _referral.REFERRAL_DETAILs.Add(rd_18);
            }
            if (_dataTable.Rows[0]["NIV"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_24 = new REFERRAL_DETAIL();
                rd_24.STD_HEALTHFACTOR_ID = 24;
                _referral.REFERRAL_DETAILs.Add(rd_24);
            }
            if (_dataTable.Rows[0]["FRIS"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_13 = new REFERRAL_DETAIL();
                rd_13.STD_HEALTHFACTOR_ID = 13;
                _referral.REFERRAL_DETAILs.Add(rd_13);
            }
            if (_dataTable.Rows[0]["FNRIS"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_10 = new REFERRAL_DETAIL();
                rd_10.STD_HEALTHFACTOR_ID = 10;
                _referral.REFERRAL_DETAILs.Add(rd_10);
            }
            if (_dataTable.Rows[0]["UFRIS"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_29 = new REFERRAL_DETAIL();
                rd_29.STD_HEALTHFACTOR_ID = 29;
                _referral.REFERRAL_DETAILs.Add(rd_29);
            }
            if (_dataTable.Rows[0]["FSTL"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_14 = new REFERRAL_DETAIL();
                rd_14.STD_HEALTHFACTOR_ID = 14;
                _referral.REFERRAL_DETAILs.Add(rd_14);
            }
            if (_dataTable.Rows[0]["FNSTL"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_11 = new REFERRAL_DETAIL();
                rd_11.STD_HEALTHFACTOR_ID = 11;
                _referral.REFERRAL_DETAILs.Add(rd_11);
            }
            if (_dataTable.Rows[0]["UFSTL"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_30 = new REFERRAL_DETAIL();
                rd_30.STD_HEALTHFACTOR_ID = 30;
                _referral.REFERRAL_DETAILs.Add(rd_30);
            }
            if (_dataTable.Rows[0]["AI"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_75 = new REFERRAL_DETAIL();
                rd_75.STD_HEALTHFACTOR_ID = 75;
                _referral.REFERRAL_DETAILs.Add(rd_75);
            }
            if (_dataTable.Rows[0]["SCI"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_79 = new REFERRAL_DETAIL();
                rd_79.STD_HEALTHFACTOR_ID = 79;
                _referral.REFERRAL_DETAILs.Add(rd_79);
            }
            if (_dataTable.Rows[0]["RST"].ToString() == "1")
            {
                REFERRAL_DETAIL rd_25 = new REFERRAL_DETAIL();
                rd_25.STD_HEALTHFACTOR_ID = 25;
                _referral.REFERRAL_DETAILs.Add(rd_25);
            }
        }
        catch (Exception ex)
        {
            Logging.WriteLogToFile("Error processing new referral", ex);
        }
    }

    private void CheckAndSetupVTABButton()
    {
        VTADataButton.Visible = _referral.PATIENT.VTA_PATIENTs.Count() > 0;

    }
    private void ReBindChangedData()
    {
        //ReloadWorkflowTypes();
        FormViewBreadCrumb.DataBind();
        if (_referral.STD_REFERRALSTS_ID == 5)
            Workflows.DataBind();

    }


    private void ShowReferralFromEntity()
    {
        LabelReferralDate.Text = String.Format("{0:d}", _referral.REFERRAL_DATE);
        TEFSCDateLabel.Text = String.Format("{0:d}", _referral.CREATED);
        LabelCreatedBy.Text = _referral.CREATEDBY;

        //hide these on new referrals
        if (_referral.STD_REFERRALSTS_ID == 4)
        {
            ReviewedDateBox.Text = String.Empty;
            ReviewedByBox.Text = String.Empty;
        }
        else
        {
            ReviewedDateBox.Text = String.Format("{0:d}", _referral.REVIEW_DATE);
            ReviewedByBox.Text = DropDownLists.ConvertNullToEmptyString(_referral.REVIEW_BY);
        }
        //VAMC related information
        VAMCContactName.Text = _referral.VAMC_CONTACT_NAME;
        VAMCContactEMail.Text = _referral.VAMC_CONTACT_EMAIL;
        VAMCContactFAX.Text = _referral.VAMC_CONTACT_FAX;
        VAMCContactPhone.Text = _referral.VAMC_CONTACT_PHONE;

        //PCP information 

        PCPEMail.Text = _referral.PROVIDER_EMAIL;
        PCPName.Text = _referral.PROVIDER_NAME;
        PCPPhone.Text = _referral.PROVIDER_PHONE;
        PCPFax.Text = _referral.PROVIDER_FAX;
        PCPAddress1.Text = _referral.PROVIDER_ADDRESS_LINE1;
        PCPAddress2.Text = _referral.PROVIDER_ADDRESS_LINE2;
        PCPAddress3.Text = _referral.PROVIDER_ADDRESS_LINE3;
        PCPCity.Text = _referral.PROVIDER_CITY;
        PCPState.Text = _referral.PROVIDER_STATE;
        PCPCountry.Text = _referral.PROVIDER_COUNTRY;
        PCPPostal.Text = _referral.PROVIDER_POSTAL_CODE;

        TextIneligibleReason.Text = _referral.COMMENT_TEXT;

        SetupHealthFactors();
    }

    private void RefreshVAMCListAndSelectInstitution()
    {
        VAMCFAcilityNamesListBox.Items.Clear();
        VAMCFAcilityNamesListBox.Items.Add(new ListItem("(Please select institution)", "0"));
        VAMCFAcilityNamesListBox.Items.AddRange(DropDownLists.GetInsitutionList().Select(kvp => new ListItem(kvp.Value, kvp.Key.ToString())).ToArray());

        if (_referral.STD_INSTITUTION_ID.HasValue && _referral.STD_INSTITUTION_ID > 0)
            VAMCFAcilityNamesListBox.SelectedIndex = VAMCFAcilityNamesListBox.Items.IndexOf(VAMCFAcilityNamesListBox.Items.FindByValue(_referral.STD_INSTITUTION_ID.GetValueOrDefault().ToString()));
        else if (_referral.STD_INSTITUTION != null)
            VAMCFAcilityNamesListBox.SelectedIndex = VAMCFAcilityNamesListBox.Items.IndexOf(VAMCFAcilityNamesListBox.Items.FindByValue(_referral.STD_INSTITUTION.ID.ToString()));
        else
            VAMCFAcilityNamesListBox.SelectedIndex = 0;
    }

    protected void AddContact_Click(object sender, EventArgs e)
    {
        Response.Redirect(string.Format("ReferralContact.aspx?referralId={0}", _referral.REFERRAL_ID));
    }

    protected void btnPushData_Click(object sender, EventArgs e)
    {
        // Call the appropriate stored procedure

        try
        {
            if (WorkingRefId.Value.Length > 0 && Convert.ToInt32(WorkingRefId.Value) > 0)
            {
                // Check to make sure this entry doesn't exist 
                var succeeded = SqlProvider.ExecuteScalar(
                    "RegistryConnectionString", UPDATE_REFERRAL, new object[] { _referral.REFERRAL_ID, WorkingRefId.Value });

                if (succeeded != DBNull.Value && Convert.ToInt32(succeeded) == 0)
                {
                    Logging.WriteLogToFile(User.Identity.Name, "Pushed referral: " + WorkingRefId.Value + " to referral: " +
                        _referral.REFERRAL_ID.ToString());

                    ClientScript.RegisterStartupScript(typeof(Page), "alertMessage",
                        "<script type='text/javascript'>alert('Working referral was successfully updated');</script>");
                }
                else
                {
                    Logging.WriteLogToFile(User.Identity.Name, "Failed pushing referral: " + WorkingRefId.Value + " to referral: " +
                        _referral.REFERRAL_ID.ToString() + " - Error Code: " + succeeded);

                    ClientScript.RegisterStartupScript(typeof(Page), "alertMessage",
                        "<script type='text/javascript'>alert('Update Failed');</script>");
                }
            }
            else
            {
                Logging.WriteLogToFile(User.Identity.Name, "Failed trying to pushed referral: " + WorkingRefId.Value + " to referral: " +
                    _referral.REFERRAL_ID.ToString() + ": Invalid duplicate referral ID");
            }
        }
        catch (Exception ex)
        {
            Logging.WriteLogToFile(User.Identity.Name, "Failed trying to pushed referral: " + WorkingRefId.Value + ": caught exception");
            Logging.WriteLogToFile(User.Identity.Name, ex);
        }
    }
    protected void ClearButton_Click(object sender, EventArgs e)
    {
        _referral.SetAsChangeTrackingRoot();
        //assign referral a "new" status 
        _referral.STD_REFERRALSTS_ID = 4;
        _referral.StatusChanged = true;
        _referral.COMMENT_TEXT = null;

        _referral.DUPLICATE_FLAG = false;
        _referralManager.Update(_referral);

        SuppressSaveSuccessMessage = true;

        getRefreshReferral(_referral.REFERRAL_ID);

        //re-load referral since we just saved it
        ShowReferralFromEntity();
        ReBindChangedData();

        SetUpNewReferral();

    }

    private void LoadIraqAfghanServiceHealthFactor(int patientId)
    {
        CheckBox checkBoxIraqAfghanService = (CheckBox)FormViewPatientInformation.FindControl("CheckBoxIraqAfghanService");

        if (checkBoxIraqAfghanService != null)
        {
            checkBoxIraqAfghanService.Checked = _referralManager.IraqAfghanServiceHealthFactorExists(patientId);
        }
    }

    private void ReLoadInstitutionAddress(bool useReferralAddress)
    {
        if (useReferralAddress)
        {
            if (String.IsNullOrEmpty(_referral.ADDRESS_LINE1))
            {
                if (String.IsNullOrEmpty(_referral.STD_INSTITUTION.MAILINGADDRESSLINE1))
                {
                    VAMCAddress1Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETADDRESSLINE1);
                    VAMCAddress2Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETADDRESSLINE2);
                    VAMCAddress3Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETADDRESSLINE3);
                    VAMCAddressCity.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETCITY);
                    VAMCAddressPostal.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETPOSTALCODE);
                    if (_referral.STD_INSTITUTION.STREETSTATE_ID.HasValue)
                    {
                        VAMCAddressState.Text = DropDownLists.ConvertNullToEmptyString(_cityStateManager.GetStateByID(_referral.STD_INSTITUTION.STREETSTATE_ID.GetValueOrDefault()));
                    }

                    if (_referral.STD_INSTITUTION.STREETCOUNTRY_ID.HasValue)
                    {
                        VAMCAddressCountry.Text = DropDownLists.ConvertNullToEmptyString(_cityStateManager.GetCountryByID(_referral.STD_INSTITUTION.STREETCOUNTRY_ID.GetValueOrDefault()));
                    }
                }
                else
                {
                    VAMCAddress1Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGADDRESSLINE1);
                    VAMCAddress2Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGADDRESSLINE2);
                    VAMCAddress3Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGADDRESSLINE3);
                    VAMCAddressCity.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGCITY);
                    VAMCAddressPostal.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGPOSTALCODE);
                    if (_referral.STD_INSTITUTION.MAILINGSTATE_ID.HasValue)
                    {
                        VAMCAddressState.Text = DropDownLists.ConvertNullToEmptyString(_cityStateManager.GetStateByID(_referral.STD_INSTITUTION.MAILINGSTATE_ID.GetValueOrDefault()));
                    }

                    if (_referral.STD_INSTITUTION.MAILINGCOUNTRY_ID.HasValue)
                    {
                        VAMCAddressCountry.Text = DropDownLists.ConvertNullToEmptyString(_cityStateManager.GetCountryByID(_referral.STD_INSTITUTION.MAILINGCOUNTRY_ID.GetValueOrDefault()));
                    }
                }
            }
            else
            {
                VAMCAddress1Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.ADDRESS_LINE1);
                VAMCAddress2Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.ADDRESS_LINE2);
                VAMCAddress3Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.ADDRESS_LINE3);
                VAMCAddressCity.Text = DropDownLists.ConvertNullToEmptyString(_referral.CITY);
                VAMCAddressPostal.Text = DropDownLists.ConvertNullToEmptyString(_referral.POSTAL_CODE);
                VAMCAddressState.Text = DropDownLists.ConvertNullToEmptyString(_referral.STATE);
                VAMCAddressCountry.Text = DropDownLists.ConvertNullToEmptyString(_referral.COUNTRY);
            }
        }
        else
        {
            if (String.IsNullOrEmpty(_referral.STD_INSTITUTION.MAILINGADDRESSLINE1))
            {
                VAMCAddress1Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETADDRESSLINE1);
                VAMCAddress2Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETADDRESSLINE2);
                VAMCAddress3Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETADDRESSLINE3);
                VAMCAddressCity.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETCITY);
                VAMCAddressPostal.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.STREETPOSTALCODE);
                if (_referral.STD_INSTITUTION.STREETSTATE_ID.HasValue)
                {
                    VAMCAddressState.Text = DropDownLists.ConvertNullToEmptyString(_cityStateManager.GetStateByID(_referral.STD_INSTITUTION.STREETSTATE_ID.GetValueOrDefault()));
                }

                if (_referral.STD_INSTITUTION.STREETCOUNTRY_ID.HasValue)
                {
                    VAMCAddressCountry.Text = DropDownLists.ConvertNullToEmptyString(_cityStateManager.GetCountryByID(_referral.STD_INSTITUTION.STREETCOUNTRY_ID.GetValueOrDefault()));
                }
            }
            else
            {
                VAMCAddress1Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGADDRESSLINE1);
                VAMCAddress2Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGADDRESSLINE2);
                VAMCAddress3Box.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGADDRESSLINE3);
                VAMCAddressCity.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGCITY);
                VAMCAddressPostal.Text = DropDownLists.ConvertNullToEmptyString(_referral.STD_INSTITUTION.MAILINGPOSTALCODE);
                if (_referral.STD_INSTITUTION.MAILINGSTATE_ID.HasValue)
                {
                    VAMCAddressState.Text = DropDownLists.ConvertNullToEmptyString(_cityStateManager.GetStateByID(_referral.STD_INSTITUTION.MAILINGSTATE_ID.GetValueOrDefault()));
                }

                if (_referral.STD_INSTITUTION.MAILINGCOUNTRY_ID.HasValue)
                {
                    VAMCAddressCountry.Text = DropDownLists.ConvertNullToEmptyString(_cityStateManager.GetCountryByID(_referral.STD_INSTITUTION.MAILINGCOUNTRY_ID.GetValueOrDefault()));
                }
            }
        }

    }

    //get user entered values back into the entity
    private void SaveReferralFromPage()
    {

        //referral.REFERRAL_DATE = Helpers.ConvertStringToNullable<DateTime>(ReferralDateBox.Text.Trim());
        _referral.VAMC_CONTACT_NAME = VAMCContactName.Text.Trim();
        _referral.VAMC_CONTACT_EMAIL = VAMCContactEMail.Text.Trim();
        _referral.VAMC_CONTACT_FAX = VAMCContactFAX.Text.Trim();
        _referral.VAMC_CONTACT_PHONE = VAMCContactPhone.Text.Trim();

        _referral.PROVIDER_EMAIL = PCPEMail.Text.Trim();
        _referral.PROVIDER_NAME = PCPName.Text.Trim();
        _referral.PROVIDER_PHONE = PCPPhone.Text.Trim();
        _referral.PROVIDER_FAX = PCPFax.Text.Trim();
        _referral.PROVIDER_ADDRESS_LINE1 = PCPAddress1.Text.Trim();
        _referral.PROVIDER_ADDRESS_LINE2 = PCPAddress2.Text.Trim();
        _referral.PROVIDER_ADDRESS_LINE3 = PCPAddress3.Text.Trim();
        _referral.PROVIDER_CITY = PCPCity.Text.Trim();
        _referral.PROVIDER_STATE = PCPState.Text.Trim();
        _referral.PROVIDER_COUNTRY = PCPCountry.Text.Trim();
        _referral.PROVIDER_POSTAL_CODE = PCPPostal.Text.Trim();



        //referral institution information 
        _referral.ADDRESS_LINE1 = VAMCAddress1Box.Text.Trim();
        _referral.ADDRESS_LINE2 = VAMCAddress2Box.Text.Trim();
        _referral.ADDRESS_LINE3 = VAMCAddress3Box.Text.Trim();
        _referral.CITY = VAMCAddressCity.Text.Trim();
        _referral.POSTAL_CODE = VAMCAddressPostal.Text.Trim();
        _referral.STATE = VAMCAddressState.Text.Trim();
        _referral.COUNTRY = VAMCAddressCountry.Text.Trim();
        if (VAMCFAcilityNamesListBox.SelectedIndex > 0)
        {
            _referral.STD_INSTITUTION_ID = Helpers.getIntFromString(VAMCFAcilityNamesListBox.SelectedValue);
        }
        else
        {
            _referral.STD_INSTITUTION_ID = null;
        }

        if (_referral.REFERRAL_ID == 0)
        {
            int patientId = 0;
            if (int.TryParse(Request.QueryString["patientId"], out patientId))
            {
                if (patientId > 0)
                {
                    _referral.PATIENT_ID = patientId;
                    _referral.REFERRAL_DATE = DateTime.Now;
                    _referral.STD_REGISTRY_ID = EFRSharedConstant.EfrRegistryId;
                    if (_referral.STD_REFERRALSTS_ID == 0)
                    {
                        _referral.STD_REFERRALSTS_ID = (int)EfrReferralStatus.New;
                    }
                    _referral.STD_REFERRALRANK_ID = EFRSharedConstant.EfrReferralRankId;
                }

            }
        }

    }

    //this is a function that sets up healthfactor area according to referral health factors
    private void SetupHealthFactors()
    {
        if (!_isOriginal)
        {
            // Set up referral details

            List<REFERRAL_DETAIL> listDetails = _referral.REFERRAL_DETAILs.GroupBy(s => s.STD_HEALTHFACTOR_ID)
                                     .Select(g => g.OrderByDescending(x => x.CREATED).First()).ToList();

            foreach (var detail in listDetails)
            {
                //Helpers.SetUpHealthFactorCheckBoxes(PanelHealthfactors, detail.STD_HEALTHFACTOR_ID);
                var lists = PanelHealthfactors.GetAllControls().OfType<CheckBoxList>();

                foreach (var list in lists)
                {
                    ListItem item = list.Items.FindByValue(detail.STD_HEALTHFACTOR_ID.ToString());
                    if (item != null)
                    {
                        //if we find factor nothing else to do here 
                        item.Selected = true;

                    }
                }

                switch (detail.STD_HEALTHFACTOR_ID)
                {
                    case 69:

                        EFRPresentBox.Text = detail.COMMENT_TEXT;
                        EFRPresentBox.Enabled = true;
                        break;

                    case 1:

                        BlastSourceOtherBox.Text = detail.COMMENT_TEXT;
                        BlastSourceOtherBox.Enabled = true;
                        break;

                    case 79:

                        CognitiveImpairmentBox.Text = detail.COMMENT_TEXT;
                        CognitiveImpairmentBox.Enabled = true;
                        break;

                    case 75:

                        AcuteIllnessBox.Text = detail.COMMENT_TEXT;
                        AcuteIllnessBox.Enabled = true;
                        break;

                    default:

                        break;
                }
            }
        }
        else
        {
            // Set referral details from Original Details information

            foreach (ListItem cb in CheckBoxListHealthFactors1.Items)
            {
                if (_dataTable.Rows[0]["EFP"].ToString() == "1" && cb.Value == "69")
                {
                    cb.Selected = true;
                    EFRPresentBox.Text = _dataTable.Rows[0]["EFP Comment"].ToString();
                }
                else if (_dataTable.Rows[0]["NEF"].ToString() == "1" && cb.Value == "70")
                    cb.Selected = true;
            }

            foreach (ListItem cb in CheckBoxListHealthFactors9.Items)
            {
                if (_dataTable.Rows[0]["BI"].ToString() == "1" && cb.Value == "4")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["NBI"].ToString() == "1" && cb.Value == "21")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["BEI"].ToString() == "1" && cb.Value == "3")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["NBEI"].ToString() == "1" && cb.Value == "20")
                    cb.Selected = true;
            }

            foreach (ListItem cb in CheckBoxListHealthFactors3.Items)
            {
                if (_dataTable.Rows[0]["IED"].ToString() == "1" && cb.Value == "17")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["RPG"].ToString() == "1" && cb.Value == "26")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["LM"].ToString() == "1" && cb.Value == "19")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["Gr"].ToString() == "1" && cb.Value == "16")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["EF"].ToString() == "1" && cb.Value == "8")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["FF"].ToString() == "1" && cb.Value == "15")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["UT"].ToString() == "1" && cb.Value == "2")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["Other"].ToString() == "1" && cb.Value == "1")
                {
                    cb.Selected = true;
                    BlastSourceOtherBox.Text = _dataTable.Rows[0]["Other COMMENT"].ToString();
                }
            }

            foreach (ListItem cb in CheckBoxListHealthFactors5.Items)
            {
                if (_dataTable.Rows[0]["FIB"].ToString() == "1" && cb.Value == "9")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["NFIB"].ToString() == "1" && cb.Value == "22")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["UFIB"].ToString() == "1" && cb.Value == "27")
                    cb.Selected = true;
            }

            foreach (ListItem cb in CheckBoxListHealthFactors4.Items)
            {
                if (_dataTable.Rows[0]["FOR"].ToString() == "1" && cb.Value == "12")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["NFOR"].ToString() == "1" && cb.Value == "23")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["UFOR"].ToString() == "1" && cb.Value == "28")
                    cb.Selected = true;
            }

            foreach (ListItem cb in CheckBoxListHealthFactors8.Items)
            {
                if (_dataTable.Rows[0]["IV"].ToString() == "1" && cb.Value == "18")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["NIV"].ToString() == "1" && cb.Value == "24")
                    cb.Selected = true;
            }

            foreach (ListItem cb in CheckBoxListHealthFactors2.Items)
            {
                if (_dataTable.Rows[0]["FRIS"].ToString() == "1" && cb.Value == "13")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["FNRIS"].ToString() == "1" && cb.Value == "10")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["UFRIS"].ToString() == "1" && cb.Value == "29")
                    cb.Selected = true;
            }

            foreach (ListItem cb in CheckBoxListHealthFactors7.Items)
            {
                if (_dataTable.Rows[0]["FSTL"].ToString() == "1" && cb.Value == "14")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["FNSTL"].ToString() == "1" && cb.Value == "11")
                    cb.Selected = true;
                else if (_dataTable.Rows[0]["UFSTL"].ToString() == "1" && cb.Value == "30")
                    cb.Selected = true;
            }

            foreach (ListItem cb in CheckBoxListHealthFactors10.Items)
            {
                if (_dataTable.Rows[0]["AI"].ToString() == "1" && cb.Value == "75")
                {
                    cb.Selected = true;
                    AcuteIllnessBox.Text = _dataTable.Rows[0]["AI COMMENT"].ToString();
                }
                else if (_dataTable.Rows[0]["SCI"].ToString() == "1" && cb.Value == "79")
                {
                    cb.Selected = true;
                    CognitiveImpairmentBox.Text = _dataTable.Rows[0]["SCI COMMENT"].ToString();
                }
                else if (_dataTable.Rows[0]["RST"].ToString() == "1" && cb.Value == "25")
                    cb.Selected = true;
            }
        }
    }

    private void SaveHealthFactorsFromPage()
    {

        //List<Control> controls = Helpers.BuildControlList(PanelHealthfactors, typeof(CheckBoxList));
        var checkBoxLists = PanelHealthfactors.GetAllControls().OfType<CheckBoxList>();

        foreach (CheckBoxList boxList in checkBoxLists)
        {
            //CheckBoxList boxList = control as CheckBoxList;

            foreach (ListItem item in boxList.Items)
            {

                int value = Convert.ToInt32(item.Value);

                if (item.Selected)
                {
                    if (value.In(_referral.REFERRAL_DETAILs.Select(rd => rd.STD_HEALTHFACTOR_ID).ToList()))
                    {
                        //do nothing except saving a comment since referral is already in 
                        //REFERRAL_DETAIL detail = _referral.REFERRAL_DETAILs.Where(rfd => rfd.STD_HEALTHFACTOR_ID == value).Single();
                        REFERRAL_DETAIL detail = _referral.REFERRAL_DETAILs.Where(rfd => rfd.STD_HEALTHFACTOR_ID == value)
                                                    .OrderByDescending(x => x.CREATED).First();

                        SetCommentForReferralDetail(ref detail, value);
                    }
                    else
                    {
                        //need to generate and save a new healthfactor, as it's not presently in a database 
                        REFERRAL_DETAIL detail = new REFERRAL_DETAIL();

                        SetCommentForReferralDetail(ref detail, value);
                        detail.REFERRAL_ID = _referral.REFERRAL_ID;
                        detail.STD_HEALTHFACTOR_ID = value;

                        _referral.REFERRAL_DETAILs.Add(detail);
                    }
                }
                else
                {
                    if (value.In(_referral.REFERRAL_DETAILs.Select(rd => rd.STD_HEALTHFACTOR_ID).ToList()))
                    {
                        //healthfactor is in a database and was unselected so need to delete it 
                        //REFERRAL_DETAIL detail = _referral.REFERRAL_DETAILs.Where(rfd => rfd.STD_HEALTHFACTOR_ID == value).Single();
                        //referral.REFERRAL_DETAILs.Remove(detail); 
                        REFERRAL_DETAIL detail = _referral.REFERRAL_DETAILs.Where(rfd => rfd.STD_HEALTHFACTOR_ID == value)
                                                    .OrderByDescending(x => x.CREATED).First();
                        detail.SetAsDeleteOnSubmit();

                    }
                    else
                    {
                        //healthfactor is not in a database so no need to do anything

                    }

                }
            }
        }
    }

    private void SetCommentForReferralDetail(ref REFERRAL_DETAIL detail, int factorId)
    {
        switch (factorId)
        {
            case 69:

                detail.COMMENT_TEXT = EFRPresentBox.Text;
                break;

            case 1:

                detail.COMMENT_TEXT = BlastSourceOtherBox.Text;
                break;

            case 79:

                detail.COMMENT_TEXT = CognitiveImpairmentBox.Text;
                break;

            case 75:

                detail.COMMENT_TEXT = AcuteIllnessBox.Text;
                break;
        }
    }

    private void CheckReferralForDuplicates()
    {
        EFR_REFERRAL duplicate = _referral.DuplicateReferral() as EFR_REFERRAL;

        if (duplicate != null)
        {
            PanelReferralDuplicate.Visible = true;
            //PanelReferralDuplicate.GroupingText = "&nbsp;Duplicate&nbsp;"; 
            LabelDuplicateHeader.Text = "Duplicate";
            DuplicateWording.Text = "This is a duplicate of a previously triaged referral. You will not be able to accept it, but may review and edit the previously triaged referral by following the link below.";

            DuplicateOriginalLink.Text = "Go to Previously Triaged Referral";
            DuplicateOriginalLink.NavigateUrl = Request.Url.AbsolutePath + "?referralId=" + duplicate.REFERRAL_ID;
            WorkingRefId.Value = duplicate.REFERRAL_ID.ToString();

            _referral.DUPLICATE_FLAG = true;
            _referral.COMMENT_TEXT = "Duplicate";

            // only show the Push button to TEFSC administrators for manually created referrals (unless otherwise configured)
            try
            {
                _userAccountManager = (UserAccountManager)HttpContext.Current.User;

                if (_userAccountManager.GetRoles().Contains(4) &&
                    (duplicate.CREATEDBY.Contains("\\") ||
                     ConfigurationManager.AppSettings["Referral_PushToEtl"].ToString().Trim() == "true") &&
                    !_isOriginal)
                {
                    btnPushData.Visible = true;
                }
            }
            catch (Exception ex)
            {
                Logging.WriteLogToFile("Could not determine user roles for showing Push button", ex);
            }

            First.Enabled = false;
            Second.Enabled = true;
            Third.Enabled = false;
            Inner1.Enabled = false;
            TextIneligibleReason.Text = "Duplicate";
        }
    }

    private void CheckAndSetupOriginalReferral()
    {
        if (Request.UrlReferrer != null)
        {
            string originalURL = Request.UrlReferrer.AbsoluteUri.ToString();
            string returnURL = originalURL;
            string leftPartOfQueryString = "referral.aspx?referralId=";
            string refId = "";

            if ((originalURL.IndexOf(leftPartOfQueryString, StringComparison.CurrentCultureIgnoreCase) > 0
                && !originalURL.Contains("patientId=")))
            {
                int paramStartLoc = originalURL.IndexOf('&');

                if (paramStartLoc > -1)   // Cut off second+ parameters
                {
                    originalURL = originalURL.Substring(0, paramStartLoc);
                }

                // get the referralId 
                int startOfRefId = originalURL.IndexOf("referralId=") + "referralId=".Length;
                refId = originalURL.Substring(startOfRefId, originalURL.Length - startOfRefId);

                PanelReferralDuplicate.Visible = true;

                if (_isOriginal)
                {
                    // We are viewing the original referral
                    PanelReferralDuplicate.Visible = true;

                    //DuplicateWording.Text = "&nbsp;";

                    if (_referral.DUPLICATE_FLAG == true)
                    {
                        LabelDuplicateHeader.Text = "Original Referral Data for Duplicate Referral";
                        DuplicateOriginalLink.Text = "Go back to Duplicate Referral";
                        DuplicateWording.Text = "&nbsp;";
                    }
                    else
                    {
                        LabelDuplicateHeader.Text = "Original Referral Data for Working Referral";
                        DuplicateOriginalLink.Text += "Go back to Working Referral";
                    }

                    DuplicateOriginalLink.NavigateUrl = returnURL; // Request.Url.AbsolutePath + "?referralId=" + refId + "&from=originalReferral";

                    // Since this is read-only view, disable all controls
                    PanelDetails.Enabled = false;
                    PanelHealthfactors.Enabled = false;
                    PanelWorkflows.Enabled = false;
                    //Buttons.Enabled = false;
                    ButtonSave.Enabled = false;
                    OKButton.Enabled = false;
                    AddContactButton.Enabled = false;
                    PanelNoActionIneligible.Enabled = false;
                    PanelChecklist.Enabled = false;
                    PanelNoActionIneligible.Enabled = false;
                }
                else if (_referral.DUPLICATE_FLAG == true)
                {
                    PanelReferralDuplicate.Visible = true;

                    //LabelDuplicateHeader.Text = "Duplicate";
                    //DuplicateWording.Text = "This is a duplicate of a previously triaged referral. You will not be able to accept it, but may review and edit the previously triaged referral by following the link below.";

                    //DuplicateOriginalLink.Text = "Go back to Previously Triaged Referral";
                    //DuplicateOriginalLink.NavigateUrl = Request.Url.AbsolutePath + "?referralId=" + refId;
                }
                else if (returnURL.Contains("workingReferral"))
                {
                    // Do not show panel if we navigated to the Original view, then are coming
                    // back to the working referral

                    PanelReferralDuplicate.Visible = false;
                }
                else
                {
                    // We navigated from the duplicate referral to the working referral
                    // Show the panel in this case

                    LabelDuplicateHeader.Text = "Previously Triaged Referral";

                    DuplicateOriginalLink.Text = "Go back to Duplicate Referral";
                    DuplicateOriginalLink.NavigateUrl = Request.Url.AbsolutePath + "?referralId=" + refId;
                }
            }
        }
    }

    public override string TopMenuName()
    {
        return "My Tasks";
    }

    public override string MenuItemName()
    {
        return "EFR_REFERRAL";
    }

    public override string AppObjectCode()
    {
        return "EFR_REFERRAL";
    }

    //private void ReloadWorkflowTypes()
    //{
    //    if (referral.STD_REFERRALSTS_ID == 5)
    //    {
    //        if (Workflows.Rows.Count > 0)
    //        {

    //            DropDownList ddlWFTypes = Workflows.FooterRow.FindControl("WorkflowTypesList") as DropDownList;

    //            ddlWFTypes.Items.Clear();
    //            ddlWFTypes.Items.Add(new ListItem("(Select One)", "0"));
    //            ddlWFTypes.Items.AddRange(DropDownLists.GetWorkflowTypeList().Select(kvp => new ListItem(kvp.Value, kvp.Key.ToString())).ToArray());
    //            ddlWFTypes.SelectedIndex = 0;
    //        }
    //        else
    //        {
    //             DropDownList ddlWFEmptyTypes = Workflows.Controls[0].Controls[0].FindControl("WorkflowTypesEmptyList") as DropDownList;

    //             ddlWFEmptyTypes.Items.Clear();
    //             ddlWFEmptyTypes.Items.Add(new ListItem("(Select One)", "0"));
    //             ddlWFEmptyTypes.Items.AddRange(DropDownLists.GetWorkflowTypeList().Select(kvp => new ListItem(kvp.Value, kvp.Key.ToString())).ToArray());
    //             ddlWFEmptyTypes.SelectedIndex = 0;
    //        }
    //    }
    //}

    protected void Workflows_RowDataBound(object sender, GridViewRowEventArgs e)
    {

        if (e.Row.RowType == DataControlRowType.EmptyDataRow)
        {
            DropDownList ddlWFEmptyTypes = Workflows.Controls[0].Controls[0].FindControl("WorkflowTypesEmptyList") as DropDownList;
            if (ddlWFEmptyTypes != null)
            {
                ddlWFEmptyTypes.Items.Clear();
                ddlWFEmptyTypes.Items.Add(new ListItem("(Select One)", "0"));
                ddlWFEmptyTypes.Items.AddRange(DropDownLists.GetWorkflowTypeList().Select(kvp => new ListItem(kvp.Value, kvp.Key.ToString())).ToArray());
                ddlWFEmptyTypes.SelectedIndex = 0;
            }

        }

        else if (e.Row.RowType == DataControlRowType.Footer)
        {
            DropDownList ddlWFTypes = e.Row.FindControl("WorkflowTypesList") as DropDownList;
            if (ddlWFTypes != null)
            {
                ddlWFTypes.Items.Clear();
                ddlWFTypes.Items.Add(new ListItem("(Select One)", "0"));
                ddlWFTypes.Items.AddRange(DropDownLists.GetWorkflowTypeList().Select(kvp => new ListItem(kvp.Value, kvp.Key.ToString())).ToArray());
                ddlWFTypes.SelectedIndex = 0;
            }
        }


    }

    protected void BioFollowUpsRepeater_ItemCommand(object source, RepeaterCommandEventArgs e)
    {

        HiddenField hiddenID = (HiddenField)e.Item.FindControl("CaseActivityIDField");
        string id = hiddenID.Value;

        int resultId;
        if (int.TryParse(id, out resultId))
        {

            if (e.CommandName.Equals("Create"))
            {
                WKF_CASE_ACTIVITY activity = _referral.WKF_CASEs.SelectMany(o => o.WKF_CASE_ACTIVITies).Where(t => t.WKF_CASE_ACTIVITY_ID == resultId).First();

                WKF_CASE caseFollowUp = new WKF_CASE();
                caseFollowUp.SetAsChangeTrackingRoot(EntityState.New);

                caseFollowUp.REFERRAL_ID = _referral.REFERRAL_ID;

                int patientId;
                if (int.TryParse(FormViewPatientInformation.DataKey.Value.ToString(), out patientId))
                {
                    caseFollowUp.PATIENT_ID = patientId;
                }

                caseFollowUp.CASE_START_DATE = DateTime.Today;
                //this is an in-process workflow
                activity.FOLLOWUP_REQUIRED_FLAG = false;
                caseFollowUp.STD_WKFCASESTS_ID =
                    (int)
                    Enum.Parse(typeof(WorkflowCaseStatus),
                               Enum.GetName(typeof(WorkflowCaseStatus), WorkflowCaseStatus.InProcess));
                //this is biological workflow, since follow ups exist only for bio type
                caseFollowUp.STD_WKFCASETYPE_ID = 3;
                caseFollowUp.CASE_NUMBER = String.Empty;

                //VS - need to re-open referral if it's already closed 
                if (_referral.STD_REFERRALSTS_ID != (int)ReferralStatus.InProcess)
                    _referral.STD_REFERRALSTS_ID = (int)ReferralStatus.InProcess;
                //referral.WKF_CASEs.Add(caseFollowUp);
                caseFollowUp.SetAsInsertOnSubmit();

                _workflowCaseManager.Update(caseFollowUp, activity);
                //_referral.WKF_CASEs.Add(caseFollowUp);

                _referralManager.Update(_referral);
                getRefreshReferral(_referral.REFERRAL_ID);

                //re-load referral since we just saved it
                ShowReferralFromEntity();

                BioFollowUpsRepeater.DataBind();
                //refresh workflows gridview
                Workflows.DataBind();
            }
            else if (e.CommandName.Equals("Decline"))
            {
                WKF_CASE_ACTIVITY activity = _referral.WKF_CASEs.SelectMany(o => o.WKF_CASE_ACTIVITies).Where(t => t.WKF_CASE_ACTIVITY_ID == resultId).First();
                activity.FOLLOWUP_DECLINED_DATE = DateTime.Today;
                activity.FOLLOWUP_REQUIRED_FLAG = false;

                _workflowCaseManager.Update(activity);

                getRefreshReferral(_referral.REFERRAL_ID);

                //re-load referral since we just saved it
                ShowReferralFromEntity();

                BioFollowUpsRepeater.DataBind();
            }
        }
    }

    protected void Workflows_RowCommand(object sender, GridViewCommandEventArgs e)
    {

        string[] arr = { "ADD", "EMPTYADD" };
        if (arr.Contains(e.CommandName))
        {
            WKF_CASE wfCase = new WKF_CASE();

            gridViewSelectField.Value = "1";

            switch (e.CommandName)
            {

                case "ADD":

                    DropDownList ddlWFTypes = Workflows.FooterRow.FindControl("WorkflowTypesList") as DropDownList;

                    int caseTypeOne;
                    if (ddlWFTypes != null)
                    {
                        if (int.TryParse(ddlWFTypes.SelectedValue, out caseTypeOne))
                            wfCase.STD_WKFCASETYPE_ID = caseTypeOne;
                    }
                    break;

                case "EMPTYADD":

                    DropDownList ddlWFEmptyTypes =
                        Workflows.Controls[0].Controls[0].FindControl("WorkflowTypesEmptyList") as DropDownList;

                    int caseType;
                    if (ddlWFEmptyTypes != null)
                    {
                        if (int.TryParse(ddlWFEmptyTypes.SelectedValue, out caseType))
                            wfCase.STD_WKFCASETYPE_ID = caseType;
                    }
                    break;

            }

            wfCase.REFERRAL_ID = _referral.REFERRAL_ID;

            int patientId;

            if (int.TryParse(FormViewPatientInformation.DataKey.Value.ToString(), out patientId))
            {
                wfCase.PATIENT_ID = patientId;
            }


            wfCase.CASE_START_DATE = DateTime.Today;
            //this is an in-process workflow

            wfCase.STD_WKFCASESTS_ID =
                (int)
                Enum.Parse(typeof(WorkflowCaseStatus),
                           Enum.GetName(typeof(WorkflowCaseStatus), WorkflowCaseStatus.InProcess));
            wfCase.CASE_NUMBER = String.Empty;

            //now add newly generated workflow to it's base referral 
            _referral.WKF_CASEs.Add(wfCase);

            if (_referral.STD_REFERRALSTS_ID != (int)ReferralStatus.InProcess)
                _referral.STD_REFERRALSTS_ID = (int)ReferralStatus.InProcess;

            //post changes to the database

            _referralManager.Update(_referral);

            SuppressSaveSuccessMessage = true;

            getRefreshReferral(_referral.REFERRAL_ID);

            //re-load referral since we just saved it
            ShowReferralFromEntity();
            ReBindChangedData();
        }
    }

    private void ResponseRedirect(string destination, string queryString)
    {
        try
        {
            string queryStringValue = string.Empty;
            if (queryString != null)
            {
                queryStringValue = AntiXssEncoder.UrlEncode(queryString);
            }

            string destinationUrl = string.Concat(destination, queryStringValue);
            string currentHost = Request.Url.Host;
            if (Helpers.IsDestinationUrlLocal(destinationUrl, currentHost))
            {
                //Response.Redirect(destinationUrl);
                Server.Transfer(destinationUrl);
            }
        }
        catch (Exception ex)
        {
            Logging.WriteLogToFile("Failure in to redirect to: - Referrals page.");
        }

    }
}
